# Copyright (C) 2025 Onyx and Iris
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# Special Thanks to OBS Discord user Penwywern for helping to improve this snippet.

"""
stop-record-delete-recording-start-new.py
This OBS Python script allows users to register a hotkey that, when pressed, will:
1. Stop the current recording (if active).
2. Delete the most recent recording file.
3. Start a new recording after a short delay.
Features:
- Registers a customizable hotkey in OBS under 'Hotkeys'.
- Ensures the hotkey only works when a recording is active.
- Handles OBS frontend events to safely delete the last recording after stopping.
- Provides logging for each major action (stop, delete, start).
Functions:
- script_description(): Returns a description of the script for OBS.
- script_load(settings): Registers the hotkey and loads its settings.
- callback(pressed): Hotkey callback to initiate the stop/delete/start sequence.
- on_event(event): Handles OBS events, deletes the last recording, and schedules the new recording.
- on_timer_tick(): Starts a new recording after a delay.
Requirements:
- OBS Studio with Python scripting support.
- Proper hotkey configuration in OBS settings.
License:
- GPL-3.0-or-later
Copyright (C) 2025 Onyx and Iris
"""

__version__ = "0.2.0"

from pathlib import Path

import obspython as obs

HOTKEY_ID = obs.OBS_INVALID_HOTKEY_ID


### SCRIPT SETUP ###
def script_description() -> str:
    """Registers a hotkey to delete the last recording and start a new one."""
    return (
        "This script registers a hotkey to delete the last recording and start a new one.\n"
        "The hotkey can be set in the OBS settings under 'Hotkeys'.\n"
        "Make sure to set the hotkey before using this script."
    )


def script_load(settings):
    """Register a hotkey to trigger the script."""
    HOTKEY_ID = obs.obs_hotkey_register_frontend(
        "record_delete_start", "Delete Last Recording and Start New", callback
    )
    hotkey_save = obs.obs_data_get_array(settings, "record_delete_start")
    obs.obs_hotkey_load(HOTKEY_ID, hotkey_save)
    obs.obs_data_array_release(hotkey_save)

    return HOTKEY_ID


### CALLBACKS ###
def callback(pressed):
    """Callback function for the hotkey."""
    if pressed:
        if not obs.obs_frontend_recording_active():
            obs.script_log(obs.LOG_WARNING, "Recording is not active.")
            return

        obs.obs_frontend_add_event_callback(on_event)
        obs.obs_frontend_recording_stop()


def on_event(event):
    """Event handler for the hotkey."""
    if event == obs.OBS_FRONTEND_EVENT_RECORDING_STOPPED:
        obs.obs_frontend_remove_event_callback(on_event)

        obs.script_log(obs.LOG_INFO, "Recording stopped.")
        resp = obs.obs_frontend_get_last_recording()
        file_pn = Path(resp)
        if file_pn.exists():
            file_pn.unlink()
            obs.script_log(obs.LOG_INFO, f"Deleted recording: {file_pn}")

        obs.timer_add(on_timer_tick, 1000)


def on_timer_tick():
    """Timer callback to start a new recording."""
    obs.remove_current_callback()

    obs.obs_frontend_recording_start()
    obs.script_log(obs.LOG_INFO, "Recording started.")
